import time
import numpy as np
import warnings
from gpaw import GPAW, FermiDirac
from gpaw.directmin.etdm_lcao import LCAOETDM


def read_data(output):
    saved_data = {}
    for i in output.splitlines():
        if i == '':
            continue
        mol = i.split()
        if 'None' in mol[1:]:
            saved_data[mol[0]] = np.array([None for _ in mol[1:]])
        else:
            saved_data[mol[0]] = np.array([float(_) for _ in mol[1:]])

    return saved_data


def set_calc(atoms, calc_args, txt, dm):
    if dm:
        calc = GPAW(**calc_args,
                    txt=txt,
                    eigensolver=LCAOETDM(matrix_exp='egdecomp-u-invar',
                                         representation='u-invar'),
                    mixer={'backend': 'no-mixing'},
                    nbands='nao',
                    occupations={'name': 'fixed-uniform'})
    else:
        calc = GPAW(**calc_args,
                    txt=txt,
                    occupations=FermiDirac(width=0.0, fixmagmom=True))
    atoms.set_calculator(calc)


def get_energy_and_iters(atoms, dm):
    t1 = time.time()
    e = atoms.get_potential_energy()
    t2 = time.time()

    if dm:
        iters = atoms.calc.wfs.eigensolver.eg_count
    else:
        iters = atoms.calc.get_number_of_iterations()

    return e, iters, t2 - t1


def compare_calculated_and_saved_results(e_diff, iters_diff,
                                         eig_string, name, dm):
    if e_diff > 1.0e-2:
        warnings.warn('Absolute difference in total energy '
                      'for ' + eig_string[dm] + ' calculation of '
                      + name + ' with respect to saved results '
                      'is %f eV'
                      % e_diff)
    if iters_diff > 3:
        warnings.warn('Absolute difference in total number of '
                      'iterations for ' + eig_string[dm] +
                      ' calculation of ' + name + ' with respect '
                      'to saved results is %d'
                      % iters_diff)


# Coordinates of liquid water configuration with 32 molecules
positions = [
    (-0.069, 0.824, -1.295), (0.786, 0.943, -0.752), (-0.414, -0.001, -0.865),
    (-0.282, -0.674, -3.822), (0.018, -0.147, -4.624),
    (-0.113, -0.080, -3.034),
    (2.253, 1.261, 0.151), (2.606, 0.638, -0.539), (2.455, 0.790, 1.019),
    (3.106, -0.276, -1.795), (2.914, 0.459, -2.386), (2.447, -1.053, -1.919),
    (6.257, -0.625, -0.626), (7.107, -1.002, -0.317), (5.526, -1.129, -0.131),
    (5.451, -1.261, -2.937), (4.585, -0.957, -2.503), (6.079, -0.919, -2.200),
    (-0.515, 3.689, 0.482), (-0.218, 3.020, -0.189), (0.046, 3.568, 1.382),
    (-0.205, 2.640, -3.337), (-1.083, 2.576, -3.771), (-0.213, 1.885, -2.680),
    (0.132, 6.301, -0.278), (1.104, 6.366, -0.068), (-0.148, 5.363, -0.112),
    (-0.505, 6.680, -3.285), (-0.674, 7.677, -3.447), (-0.965, 6.278, -2.517),
    (4.063, 3.342, -0.474), (4.950, 2.912, -0.663), (3.484, 2.619, -0.125),
    (2.575, 2.404, -3.170), (1.694, 2.841, -3.296), (3.049, 2.956, -2.503),
    (6.666, 2.030, -0.815), (7.476, 2.277, -0.316), (6.473, 1.064, -0.651),
    (6.860, 2.591, -3.584), (6.928, 3.530, -3.176), (6.978, 2.097, -2.754),
    (2.931, 6.022, -0.243), (3.732, 6.562, -0.004), (3.226, 5.115, -0.404),
    (2.291, 7.140, -2.455), (1.317, 6.937, -2.532), (2.586, 6.574, -1.669),
    (6.843, 5.460, 1.065), (7.803, 5.290, 0.852), (6.727, 5.424, 2.062),
    (6.896, 4.784, -2.130), (6.191, 5.238, -2.702), (6.463, 4.665, -1.259),
    (0.398, 0.691, 4.098), (0.047, 1.567, 3.807), (1.268, 0.490, 3.632),
    (2.687, 0.272, 2.641), (3.078, 1.126, 3.027), (3.376, -0.501, 2.793),
    (6.002, -0.525, 4.002), (6.152, 0.405, 3.660), (5.987, -0.447, 4.980),
    (0.649, 3.541, 2.897), (0.245, 4.301, 3.459), (1.638, 3.457, 3.084),
    (-0.075, 5.662, 4.233), (-0.182, 6.512, 3.776), (-0.241, 5.961, 5.212),
    (3.243, 2.585, 3.878), (3.110, 2.343, 4.817), (4.262, 2.718, 3.780),
    (5.942, 2.582, 3.712), (6.250, 3.500, 3.566), (6.379, 2.564, 4.636),
    (2.686, 5.638, 5.164), (1.781, 5.472, 4.698), (2.454, 6.286, 5.887),
    (6.744, 5.276, 3.826), (6.238, 5.608, 4.632), (7.707, 5.258, 4.110),
    (8.573, 8.472, 0.407), (9.069, 7.656, 0.067), (8.472, 8.425, 1.397),
    (8.758, 8.245, 2.989), (9.294, 9.091, 3.172), (7.906, 8.527, 3.373),
    (4.006, 7.734, 3.021), (4.685, 8.238, 3.547), (3.468, 7.158, 3.624),
    (5.281, 6.089, 6.035), (5.131, 7.033, 6.378), (4.428, 5.704, 5.720),
    (5.067, 7.323, 0.662), (5.785, 6.667, 0.703), (4.718, 7.252, 1.585)
]

# Results (total energy, number of iterations) of LCAO calculations on
# liquid water configurations. Obtained with wm_dm_vs_scf.py.
# 0: scf
# 1: dm
wm_saved_results = {0: np.array([[-449.2501666690716, 22],
                                 [-899.7732083940263, 21],
                                 [-1802.1232238298205, 21]]),
                    1: np.array([[-449.29433888653887, 15],
                                 [-899.8689779482846, 15],
                                 [-1802.1980642103324, 15]])}

# Results (total energy, number of iterations) of calculations
# on the molecules from the G2 set using direct minimization
# in LCAO. Obtained with g2_dm_ui_vs_scf.py.
data_g2_dm = \
    """
PH3 10 -15.067594250491323
P2 8 -8.379318635747365
CH3CHO 17 -37.75204876792297
H2COH 16 -23.69898686685713
CS 13 -9.417455952101529
OCHCHO 15 -35.666514372774785
C3H9C 15 -66.59136951997174
CH3COF 16 -38.4905594550757
CH3CH2OCH3 14 -61.30296354126676
HCOOH 16 -28.408002357246865
HCCl3 14 -17.40322721170416
HOCl 19 -9.763721177128842
H2 6 -6.490747304345913
SH2 10 -10.506039275136805
C2H2 10 -21.831120179124515
C4H4NH 15 -60.36706614168592
CH3SCH3 13 -42.75640371331927
SiH2_s3B1d 11 -8.439696476805787
CH3SH 12 -26.575973842997186
CH3CO 17 -32.72729035439991
CO 11 -13.815908836466543
ClF3 18 -7.279906905738146
SiH4 9 -18.407226340441163
C2H6CHOH 15 -61.72625777640576
CH2NHCH2 15 -42.17124083425262
isobutene 16 -63.67223781609167
HCO 20 -16.124831216748227
bicyclobutane 14 -54.503393011455486
LiF 14 -5.817805188634345
C2H6 12 -39.5282496643079
CN 18 -12.014894676426977
ClNO 20 -13.63804219450336
SiF4 11 -25.20937860100401
H3CNH2 14 -34.54290009661039
methylenecyclopropane 15 -54.6879018075723
CH3CH2OH 15 -45.38183883038578
NaCl 15 -3.9915002448726358
CH3Cl 13 -21.364341391719453
CH3SiH3 12 -34.95738850941009
AlF3 12 -18.708233835513557
C2H3 15 -24.91262778147432
ClF 13 -3.083951458122166
PF3 14 -17.6200414942034
PH2 11 -10.333093382175694
CH3CN 16 -35.40005851345955
cyclobutene 13 -55.09356789335748
CH3ONO 18 -37.48477835032511
SiH3 11 -13.414712732958558
C3H6_D3h 12 -47.16406379242271
CO2 10 -21.686869966483524
NO 47 -11.327631125222853
trans-butane 13 -71.89450752855547
H2CCHCl 15 -29.238625168100576
LiH 11 -3.531257198110949
NH2 13 -12.766498080792353
CH 26 -5.674378327614047
CH2OCH2 15 -36.76143971942084
C6H6 12 -73.85217535506382
CH3CONH2 18 -50.16888351727816
cyclobutane 12 -63.53872939601613
H2CCHCN 18 -43.162802485831776
butadiene 14 -55.3102844458418
H2CO 14 -21.136756510681643
CH3COOH 17 -44.92291892250132
HCF3 14 -23.33193072036996
CH3S 13 -21.690397937558533
CS2 11 -15.408202617521624
SiH2_s1A1d 10 -9.141455540051073
C4H4S 16 -52.33896491532939
N2H4 14 -29.04826077564009
OH 23 -6.982508936175612
CH3OCH3 16 -44.99441429004591
C5H5N 16 -69.10577872701126
H2O 12 -13.248879020254805
HCl 9 -5.362280651746735
CH2_s1A1d 11 -10.757688251327073
CH3CH2SH 15 -42.77182919987737
CH3NO2 16 -37.59598912842743
BCl3 12 -14.568120361655941
C4H4O 16 -54.632472747679074
CH3O 14 -23.459267720379835
CH3OH 14 -29.060866334893923
C3H7Cl 15 -53.83123625718132
isobutane 13 -71.94266082590747
CCl4 12 -15.275852686988774
CH3CH2O 16 -39.641252099285346
H2CCHF 15 -30.904670428100868
C3H7 15 -50.22805517183567
CH3 11 -17.49384850623045
O3 16 -12.433529583245473
C2H4 11 -30.958765485095768
NCCN 12 -30.658037813517044
S2 8 -6.219282082506314
AlCl3 13 -12.98573282335161
SiCl4 13 -16.78447293181344
SiO 15 -10.262784811760204
C3H4_D2d 15 -38.56777011081724
COF2 15 -22.14128628537746
2-butyne 14 -54.939470685326455
C2H5 14 -33.8500443324274
BF3 11 -21.572371839064136
N2O 14 -20.102195376334976
F2O 15 -7.399576680971737
SO2 13 -14.870794217160935
H2CCl2 14 -19.399637762862064
CF3CN 16 -34.906808720109986
HCN 14 -18.73012010449129
C2H6NH 15 -50.52630869093685
OCS 14 -18.601453536910835
ClO 52 -4.6936828818312275
C3H8 13 -55.717607762841126
HF 11 -7.0029471215133015
O2 8 -9.108226430211426
SO 14 -7.920249959323364
NH 11 -7.489024231960305
C2F4 13 -30.24116221475319
NF3 15 -14.064869244897391
CH2_s3B1d 12 -11.464140729402992
CH3CH2Cl 15 -37.64619843915724
CH3COCl 18 -36.57929939296139
NH3 12 -18.687024847371475
C3H9N 15 -66.57679757132048
CF4 12 -23.61462611454026
C3H6_Cs 15 -47.320065716714126
Si2H6 10 -30.15440392023656
HCOOCH3 16 -44.2664512112896
CCH 16 -14.799136984583923
Si2 9 -4.793225261681442
C2H6SO 17 -48.02622324542428
C5H8 13 -70.91759596894674
H2CF2 13 -22.97018289970631
Li2 7 -1.3181686820255385
CH2SCH2 13 -34.71535928912935
C2Cl4 14 -23.59447986714585
C3H4_C3v 16 -38.41646304753126
CH3COCH3 19 -54.26543917872743
F2 9 -2.7777443960323573
CH4 11 -23.399963889551103
SH 41 -5.401168291262951
H2CCO 15 -29.572163404216557
CH3CH2NH2 14 -50.82292789668282
N2 8 -15.608612050282868
Cl2 9 -2.535359129835213
H2O2 14 -16.95818938312642
Na2 10 -1.064315921432062
BeH 10 -2.9158725067581828
C3H4_C2v 14 -37.68429215491533
NO2 15 -17.169478949870694
"""

# Results (total energy, number of iterations) of calculations
# on the molecules from the G2 set using the standard SCF eigensolver
# in LCAO . Obtained with g2_dm_ui_vs_scf.py.
data_g2_scf = \
    """
PH3 18 -15.067939739162565
P2 14 -8.379364578606403
CH3CHO 17 -37.752640153790544
H2COH 18 -23.700059799196794
CS 15 -9.417393294333172
OCHCHO 17 -35.66811875892615
C3H9C 19 -66.59205808003622
CH3COF 17 -38.4908567580481
CH3CH2OCH3 17 -61.30320737242064
HCOOH 18 -28.40857295188662
HCCl3 16 -17.403156748455167
HOCl 16 -9.765718189549954
H2 12 -6.490910413647257
SH2 18 -10.506376963353159
C2H2 15 -21.8315594881508
C4H4NH 17 -60.36737073551913
CH3SCH3 18 -42.75755391907749
SiH2_s3B1d 15 -8.439652470852263
CH3SH 19 -26.57600571342069
CH3CO 15 -32.727526205821974
CO 14 -13.817286401313696
ClF3 15 -7.280418158543331
SiH4 16 -18.406906983682326
C2H6CHOH 16 -61.727709924460434
CH2NHCH2 17 -42.170691521207104
isobutene 17 -63.67196360060958
HCO 17 -16.12661908530326
bicyclobutane 15 -54.50318222862183
LiF 18 -5.818530975164816
C2H6 18 -39.52761899945607
CN 17 -12.013574497449207
ClNO 18 -13.637173578862363
SiF4 15 -25.21531130705909
H3CNH2 15 -34.542907090515556
methylenecyclopropane 17 -54.687880878011434
CH3CH2OH 15 -45.38267490450617
NaCl 16 -3.991481648956574
CH3Cl 18 -21.364671136168802
CH3SiH3 18 -34.957867383396355
AlF3 15 -18.70969513486594
C2H3 20 -24.912867583547744
ClF 15 -3.0836445438877313
PF3 16 -17.620133982392183
PH2 20 -10.333156669751412
CH3CN 17 -35.399551652561215
cyclobutene 15 -55.09416925492654
CH3ONO 22 -37.484991962456576
SiH3 19 -13.414822942888721
C3H6_D3h 15 -47.164632312754364
CO2 14 -21.690591409424943
NO 75 -11.327965166239705
trans-butane 17 -71.89452926449687
H2CCHCl 19 -29.238622363016635
LiH 17 -3.5313140027080223
NH2 13 -12.76557469532906
CH 101 -5.673636310132295
CH2OCH2 18 -36.76395515833223
C6H6 15 -73.85339953604031
CH3CONH2 19 -50.168313481824775
cyclobutane 16 -63.539274905937184
H2CCHCN 18 -43.1621120373009
butadiene 19 -55.31058655984656
H2CO 15 -21.139006626605422
CH3COOH 19 -44.924174089136265
HCF3 16 -23.33279099725403
CH3S 17 -21.690728087340183
CS2 17 -15.408150740029571
SiH2_s1A1d 19 -9.141474384102668
C4H4S 19 -52.339132780598575
N2H4 15 -29.04650645552515
OH 92 -6.984028868322686
CH3OCH3 16 -44.99641117875515
C5H5N 19 -69.10427719659162
H2O 16 -13.250986473888151
HCl 16 -5.36259443336188
CH2_s1A1d 17 -10.75750241996795
CH3CH2SH 18 -42.77177811221571
CH3NO2 17 -37.59617685073973
BCl3 15 -14.567528269012527
C4H4O 18 -54.63443308676421
CH3O 17 -23.46123621228536
CH3OH 15 -29.061696621373784
C3H7Cl 18 -53.83150398571328
isobutane 18 -71.94286533151197
CCl4 15 -15.275216282079997
CH3CH2O 17 -39.64358031935646
H2CCHF 16 -30.90462662415892
C3H7 19 -50.22810186101624
CH3 15 -17.49423441816605
O3 17 -12.436218920717389
C2H4 15 -30.960041413667845
NCCN 17 -30.656829904791742
S2 17 -6.219212745370212
AlCl3 15 -12.985497496691243
SiCl4 16 -16.784060452785397
SiO 17 -10.264395373463588
C3H4_D2d 16 -38.567667331353746
COF2 16 -22.143807001114947
2-butyne 19 -54.940138451098214
C2H5 17 -33.851029358002215
BF3 14 -21.57253092881569
N2O 19 -20.101453880304287
F2O 15 -7.401790642950211
SO2 15 -14.870076420824926
H2CCl2 19 -19.399891835301172
CF3CN 19 -34.90747376810633
HCN 17 -18.729247544127606
C2H6NH 15 -50.52638663455311
OCS 17 -18.602824279134886
ClO 168 -4.694492551841379
C3H8 18 -55.71715479135624
HF 15 -7.004240357122936
O2 15 -9.111654239498307
SO 35 -7.921946288251748
NH 16 -7.488300012131835
C2F4 14 -30.241856053715637
NF3 15 -14.066501151063182
CH2_s3B1d 16 -11.46398769462341
CH3CH2Cl 18 -37.64667833838129
CH3COCl 17 -36.58066815772619
NH3 13 -18.686996789673476
C3H9N 16 -66.5765684297463
CF4 17 -23.614422687205757
C3H6_Cs 18 -47.31966084193544
Si2H6 15 -30.154437351754027
HCOOCH3 19 -44.26973461011855
CCH 16 -14.799409011958037
Si2 17 -4.793115114511537
C2H6SO 18 -48.02735877864575
C5H8 15 -70.91659810523136
H2CF2 16 -22.971190008076956
Li2 21 -1.3181781742775984
CH2SCH2 20 -34.71520519895414
C2Cl4 17 -23.595139969720563
C3H4_C3v 16 -38.416472047250096
CH3COCH3 17 -54.26678298560979
F2 16 -2.7789800209690525
CH4 15 -23.40050852386761
SH 52 -5.401240463979583
H2CCO 15 -29.57475371469119
CH3CH2NH2 16 -50.82201231180034
N2 18 -15.60692193925049
Cl2 15 -2.535145846065773
H2O2 17 -16.958825223285647
Na2 11 -1.0643211358816327
BeH 45 -2.9158429067846203
C3H4_C2v 16 -37.684405607781144
NO2 16 -17.167930472676215
"""
